import { body, validationResult } from 'express-validator';

// Handle validation errors
export const handleValidationErrors = (req, res, next) => {
  const errors = validationResult(req);
  if (!errors.isEmpty()) {
    return res.status(400).json({
      success: false,
      message: 'Validation failed',
      errors: errors.array()
    });
  }
  next();
};

// Payment validation rules
// Note: Amount validation is handled in the controller since it depends on payment item amount
export const validatePayment = [
  body('payerName')
    .trim()
    .isLength({ min: 2, max: 100 })
    .withMessage('Payer name must be between 2 and 100 characters'),
  
  body('payerEmail')
    .isEmail()
    .normalizeEmail()
    .withMessage('Valid email is required'),
  
  body('payerPhone')
    .isMobilePhone('en-NG')
    .withMessage('Valid Nigerian phone number is required'),
  
  body('payment_item_id')
    .notEmpty()
    .withMessage('Payment item ID is required'),
  
  body('amount')
    .isFloat({ min: 0 })
    .withMessage('Amount must be a positive number'),
  
  handleValidationErrors
];

// Admin user validation rules (for creation)
export const validateAdminUser = [
  body('staff_email')
    .isEmail()
    .normalizeEmail()
    .withMessage('Valid email is required'),
  
  body('staff_id')
    .trim()
    .isLength({ min: 3, max: 50 })
    .withMessage('Staff ID must be between 3 and 50 characters'),
  
  body('surname')
    .trim()
    .isLength({ min: 2, max: 50 })
    .withMessage('Surname must be between 2 and 50 characters'),
  
  body('first_name')
    .trim()
    .isLength({ min: 2, max: 50 })
    .withMessage('First name must be between 2 and 50 characters'),
  
  body('role')
    .isIn(['bursary', 'imtc'])
    .withMessage('Role must be either bursary or imtc'),
  
  body('password')
    .optional()
    .isLength({ min: 6 })
    .withMessage('Password must be at least 6 characters'),
  
  handleValidationErrors
];

// Admin user validation rules (for updates - all fields optional)
export const validateAdminUserUpdate = [
  body('staff_email')
    .optional()
    .isEmail()
    .normalizeEmail()
    .withMessage('Valid email is required'),
  
  body('staff_id')
    .optional()
    .trim()
    .isLength({ min: 3, max: 50 })
    .withMessage('Staff ID must be between 3 and 50 characters'),
  
  body('surname')
    .optional()
    .trim()
    .isLength({ min: 2, max: 50 })
    .withMessage('Surname must be between 2 and 50 characters'),
  
  body('first_name')
    .optional()
    .trim()
    .isLength({ min: 2, max: 50 })
    .withMessage('First name must be between 2 and 50 characters'),
  
  body('other_name')
    .optional()
    .trim()
    .isLength({ max: 50 })
    .withMessage('Other name must be less than 50 characters'),
  
  body('role')
    .optional()
    .isIn(['bursary', 'imtc'])
    .withMessage('Role must be either bursary or imtc'),
  
  body('password')
    .optional()
    .isLength({ min: 6 })
    .withMessage('Password must be at least 6 characters'),
  
  body('staff_passport')
    .optional()
    .custom((value) => {
      if (!value || value.trim() === '') return true; // Allow empty strings
      // Allow both URLs and file paths
      return /^(https?:\/\/.+|\/uploads\/.+)/.test(value);
    })
    .withMessage('Staff passport must be a valid URL or file path'),
  
  handleValidationErrors
];

// Login validation rules
export const validateLogin = [
  body('email')
    .isEmail()
    .normalizeEmail()
    .withMessage('Valid email is required'),
  
  body('password')
    .notEmpty()
    .withMessage('Password is required'),
  
  handleValidationErrors
];

// Payment item validation rules
export const validatePaymentItem = [
  body('payment_item_id')
    .trim()
    .isLength({ min: 3, max: 50 })
    .withMessage('Payment item ID must be between 3 and 50 characters'),
  
  body('payment_item')
    .trim()
    .isLength({ min: 2, max: 100 })
    .withMessage('Payment item name must be between 2 and 100 characters'),
  
  body('amount')
    .isFloat({ min: 1 })
    .withMessage('Amount must be a positive number'),
  
  body('service_type_id')
    .trim()
    .notEmpty()
    .withMessage('Service type ID is required'),
  
  handleValidationErrors
];
